<?php

class WC_Custom_Status_Plugin {

    /**
     * Hooks the plugin's functions
     */
    function __construct() {

        global $woocommerce;

        // Adds the Status and Actions to the menu
        add_action('admin_menu',array(&$this,'add_admin_pages'));
        // Custom work when an order status changes to one of our statuses
        add_action('woocommerce_order_status_changed',array(&$this,'process_order_status'),0,3);
        // Add columns to the orders list
        add_filter('manage_shop_order_posts_custom_column', array(&$this,'payment_column_manage'),11);
        add_filter('woocommerce_admin_order_actions',array(&$this,'action_column_manage'),11,2);
        add_filter( 'woocommerce_general_settings', array( &$this, 'add_general_settings_option' ) );

        // Handles ajax request for deleting statuses
        add_action( 'wp_ajax_delete_custom_status', array( &$this, 'woocommerce_status_actions_delete') );


        // Cancel button to appear on custom statuses that only have the cancel checkbox checked
        add_filter( 'woocommerce_valid_order_statuses_for_cancel', array( &$this, 'cancel_button_handler' ) );


        add_filter( 'woocommerce_reports_order_statuses', array( &$this, 'display_in_reports_handler' ) );

    }

    /**
     * Allow cancel button to appear beside orders that have custom status if administrator chose that.
     * @param  array $a the statuses names that allow the cancellation of the order
     * @return array    modified array (or original if not modified)
     */
    function cancel_button_handler( $a ) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'woocommerce_order_status_action';
        /* -- Preparing your query -- */
        $query = "SELECT * FROM {$table_name}";
        $records = $wpdb->get_results( $query );

        foreach( $records as $r ) {
            if( !empty( $r->can_cancel ) && $r->can_cancel == '1' ) {
                $a[] = $r->status_name;
            }
        }
        return $a;
    }

    public function display_in_reports_handler($statuses){
         global $wpdb;
        $table_name = $wpdb->prefix . 'woocommerce_order_status_action';
        /* -- Preparing your query -- */
        $query = "SELECT new_status FROM {$table_name} WHERE display_in_reports = '1'";
        $records = $wpdb->get_results( $query );
        $a = array();
        foreach( $records as $r ) {
            if( !empty( $r->new_status ) ) {
                $a[] = $r->new_status;
            }
        }
        $b = array_merge($statuses, $a);
        return $b;
    }

    /**
     * Delete status(es) - handles the ajax request
     * @return void
     */
    function woocommerce_status_actions_delete() {
        global $woocommerce;
        // If our required post parameters are set
        if( isset( $_POST['delete_status_id'] ) || isset( $_POST['status_id'] ) ) {
            // If not an administrator disallow it
            if ( !is_admin() ) { echo '0'; die(); }
            // If user can not edit shop orders disallow it
            if ( !current_user_can('edit_shop_orders') ) { echo '0'; die(); }
            // If nonce is not set or wrong, disallow it
            if( empty( $_POST['nonce'] ) || !wp_verify_nonce( $_POST['nonce'], 'delete_custom_status' ) ) { echo '0'; die(); }
            // If we're deleting a single status and the alternative term is set
            if( ! empty( $_POST['delete_status_id'] ) && ! empty( $_POST['alt_term'] ) ) { // single deletion
                // We're expecting ids - check for valid numbers
                if( is_numeric( $_POST['delete_status_id'] ) && is_numeric( $_POST['alt_term'] ) ) {
                    // If the status id to delete is the same as the alternative term, disallow it
                    if( $_POST['delete_status_id'] == $_POST['alt_term'] ) {
                        echo '-1';
                        die();
                    }
                    // Ok Do it - delete the status
                    $woocommerce_status_actions = new WC_Custom_Status();
                    $woocommerce_status_actions->term_id = $_POST['delete_status_id'];
                    $deleted = $woocommerce_status_actions->delete();
                    if( !$deleted ) {
                        echo '0';
                    } else {
                        // Success
                        echo '1';
                    }
                    die();
                }
            } else if( !empty( $_POST['status_id'] ) ) {
                // We get a serialized form submission, parse it into params
                $params = array();
                parse_str( $_POST['status_id'], $params );
                // The array is status_id_to_delete => alternative_term_or_status_to_set_the_orders_to
                foreach( $params['status_id'] as $status_id => $alt_term ) {
                    // Ok Do it for every submitted term for deletion
                    $woocommerce_status_actions = new WC_Custom_Status();
                    $woocommerce_status_actions->term_id = $status_id;
                    $deleted = $woocommerce_status_actions->delete( $alt_term );
                    if( !$deleted ) {
                        echo '0';
                        die();
                    }
                }
                echo '1'; // if we reached here, everything was deleted successfully!
                die();
            }
            echo '0';
            die();
        }
    }

    /**
     * Adds our admin pages and the status and actions menu to the menu
     */
    function add_admin_pages() {
        //Options-writing.php is given as parent for pages that we do not want to display in the menu
        add_submenu_page('options-writing.php', __('Add New Status', 'woocommerce_status_actions'),  __('Add Status', 'woocommerce_status_actions') , 'manage_woocommerce', 'woocommerce_status_and_actions&tab=add_status', array(&$this,'add_status_page'));
        add_submenu_page('options-writing.php', __('Custom Status', 'woocommerce_status_actions'),  __('Custom Status', 'woocommerce_status_actions') , 'manage_woocommerce', 'woocommerce_status_and_actions&tab=list_status', array(&$this,'list_status'));
        add_submenu_page('options-writing.php', __('Edit Status', 'woocommerce_status_actions'),  __('Edit Status', 'woocommerce_status_actions') , 'manage_woocommerce', 'woocommerce_status_and_actions&tab=edit_status', array(&$this,'edit_status'));
        add_submenu_page('options-writing.php', __('Delete Status', 'woocommerce_status_actions'),  __('Delete Status', 'woocommerce_status_actions') , 'manage_woocommerce', 'woocommerce_status_and_actions&tab=delete_status', array(&$this,'delete_status'));
        add_submenu_page('woocommerce', __('Status & Actions', 'woocommerce_status_actions'),  __('Status & Actions', 'woocommerce_status_actions') , 'manage_woocommerce', 'woocommerce_status_and_actions', array(&$this,'create_settings_page'));
    }

    /**
     * Decides which page to render
     * @return void
     */
    public function create_settings_page() {
        global $current_tab, $current_section;

        $current = $current_section ? '' : 'class="current"';

        // If no tab, display the statuses list
        if( empty( $_GET['tab'] ) ) {
            $this->list_status();
        } else {
            switch( $_GET['tab'] ) {
                case 'list_status':
                    $this->list_status();
                    break;
                case 'add_status':
                    $this->add_status_page();
                    break;
                case 'edit_status':
                    $this->edit_status();
                    break;
                case 'delete_status':
                    $this->delete_status();
                    break;
                default:
                    $this->list_status();
                    break;
            }
        }

    }

    /**
     * Renders the delete page on bulk action and per single delete
     * @return [type] [description]
     */
    function delete_status() {
        // If status_id is not submitted view an error message
        if( ( empty( $_GET['status_id'] ) && empty( $_POST['status_id'] ) ) ) {
            ?>
            <div id='message' class='updated below-h2'>
                <p>
                    Something is wrong. Please go back and try again.
                </p>
            </div>
            <?php
        } else {
            // If the get parameter is set, it's a single deletion
            $arr = array();
            if( isset( $_GET['status_id'] ) ) {
                $arr[] = $_GET['status_id'];
                // Important for ajax handling - handled in custom-status-action-delete.js file
                $class = 'linen_confirm';
            } else {
                $arr = $_POST['status_id'];
                // Important for ajax handling
                $class = 'linen_confirm_multi';
            }
            $order_statuses = get_terms( 'shop_order_status', array( 'hide_empty' => false ) );
            // Build the selectbox and the confirmation button
            $order_select = '';
            foreach( $order_statuses as $o ) {
                $order_select .= '<option value="' . $o->term_id . '">' . $o->name . '</option>';
            }
            ?>
            <div class="wrap">
                <div class="icon32"><img src="<?php echo plugins_url( 'assets/img/status-actions-icon.png', dirname( __FILE__ ) ); ?>" width="29" height="29" /></div><h2>Delete Status</h2>
                <form action="POST" onsubmit="return false;">
            <?php
            foreach( $arr as $status_id ) {
                // Get the order status
                $t = get_term_by( 'id', $status_id, 'shop_order_status' );
                ?>
                <div id="wrapper_for_<?php echo $status_id; ?>">
                    <fieldset><p><legend>What should be done with orders that currently have the <strong><?php echo $t->name; ?></strong> status?</legend></p>
                    <ul style="list-style:none;">
                        <li>
                            <label for="delete_option1">Attribute all orders to the following status:</label>
                            <div id="select_for_<?php echo $status_id; ?>">
                                <select name="status_id[<?php echo $status_id; ?>]" class="linen_order_statuses" data-id="<?php echo $status_id; ?>">
                                    <?php echo $order_select; ?>
                                </select>
                            </div>
                        </li>
                    </ul>
                    </fieldset>
                    </p>
                    </div>
                <?php
            }
        ?>
        <p class="submit"><a data-url="<?php echo admin_url('admin.php?page=woocommerce_status_and_actions&success_add=true'); ?>" data-nonce="<?php echo wp_create_nonce('delete_custom_status'); ?>" class="button <?php echo $class; ?>" href="javascript:;">Confirm Deletion</a>
        </form>
        </div>
        <?php
        }
    }

    /**
     * Rendering the add status page
     */
    function add_status_page() {
        global $woocommerce;
         $display = '
<div class="wrap">
    <div class="icon32"><img src="' . plugins_url( 'assets/img/status-actions-icon.png', dirname( __FILE__ ) ) . '" width="29" height="29" /></div><h2>Add New Status</h2>
    <div class="error" style="display: none;">
        <p></p>
    </div>';

        //Logic for handling form submission
        if(isset($_POST['submit'])){
            try {
                // Add the status
                $woocommerce_status_actions = new WC_Custom_Status();
                $result = $woocommerce_status_actions->add_status($_POST);
                if($result) {
                    // Redirect on success
                    ?>
                    <script type="text/javascript">
                        window.location = '<?php echo admin_url('admin.php?page=woocommerce_status_and_actions&tab=list_status&success_add=true'); ?>';
                    </script>
                    <?php
                }
            } catch (RequiredFieldMissingException $e){
                $message = __('<h3>Required fields</h3> ','woocommerce_status_actions');
                $message .= '<ol>';
                foreach( $e->get_missing_fields() as $field ) {
                    $message .= '<li> The field <strong><i>' . WC_Custom_Status_Plugin::get_field_name($field) . '</i></strong> is required.</li>';
                }
                $message .= '</ol>';
                $display .= "<div class='error'>{$message}</div>";
            } catch (WPErrorException $e){
                $message = implode(',',$e->get_error_messages());
                $display .= "<div class='error'>{$message}</div>";
            } catch (Exception $e){
                $message = $e->getMessage();
                $display .= "<div class='error'>{$message}</div>";
            }
        }

        $status_color_type_html = '';
        $status_color_types = array(
                '1' => '<span class="old_status_icon status_edit woocommerce_status_font_icon woocommerce_status_actions_e03c"></span>',
                '0' => '<span  class="new_status_icon status_edit woocommerce_status_font_icon woocommerce_status_actions_e03c"></span>',
                '2' => '<strong class="status_text status_edit">preview</strong>',
                '3' => '<strong class="status_line_text status_edit">preview</strong>'
                );
        foreach($status_color_types as $sc_key => $sc_val){
            $status_color_type_html .= "<label for='radio_$sc_key'><input type='radio' name='status_color_type' id='radio_$sc_key' value='$sc_key'> $sc_val</label>";
        }

         $custom_status_list = WC_Custom_Status::get_status_list();
         $status_list = array_merge($custom_status_list,WC_Custom_Status::$default_status_list);
         // For font awesome selection
         add_thickbox();
        //Logic for displaying form
       $display .= '
    <form id="add_custom_status_form" method="POST" action="">
    <h3>Status Settings</h3>
    <p>The following fields allow you to configure the status to determine what happens when the status is applied to the order.</p>
        <table class="form-table">
            <tbody>
            <tr valign="top">
                <th scope="row"><label for="status_name">Status Name <span class="description">(required)</span> </label></th>
                <td><input type="text" name="status_name" id="status_name"><p class="description">Enter the label of the status which you would like to add</p></td>
            </tr>
            <tr valign="top">
            <th scope="row"><label for="status_icon">Status Icon </label></th>
            <td>';


            ob_start();
            ?>
            <span style="padding: 0px 10px;" class="status_font_colour default-status-icon-selector woocommerce_status_font_icon woocommerce_status_actions_e03c" href="javascript:;"></span>
            <input type="hidden" name="status_icon" value="e03c" />
            <?php
            $display .= ob_get_clean();

            $display .= '
                <a href="#TB_inline?width=600&height=550&inlineId=status-icons-list" class="thickbox">Click here to change the icon</a>
                <input type="hidden" name="status_icon" value="alarm.png" />
                <p class="description">Select an icon you would like to display as the Status icon.</p>
            </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="status_color">Status Colour <span class="description">(required)</span> </label></th>
                <td><input type="text" name="status_color" id="status_color" data-default-color="#effeff" ><p class="description">Select the main colour of your custom status.</p></td>
            </tr>
            <tr valign="top">
                <th scope="row">
                  <label for="status_color_type">Status Style <span class="description">(required)</span></label>
                  <img class="help_tip pull_right" data-tip="Select a style for your custom status." src="' . $woocommerce->plugin_url() . '/assets/images/help.png" height="16" width="16" />
                </th>
                <td>
                        '.$status_color_type_html.'
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="sends_email">Email Notification </label></th>
                <td><input type="checkbox" name="sends_email" id="sends_email" value="yes"><p class="description">Choose whether you would like to enable email notification when
this custom status is applied.</p></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="email_to">Recipient(s) <span class="description">(required)</span> </label>
                    <img class="help_tip pull_right" data-tip="Choose to whom you want the email to be sent to when the custom status is applied." src="' . $woocommerce->plugin_url() . '/assets/images/help.png" height="16" width="16" />
                </th>
                <td>
                    <select name="email_to" id="email_to">
                        <option value="customer">Customer</option>
                        <option value="admin">Admin</option>
                        <option value="both">Customer and Admin</option>
                        <option value="custom">Custom Email Address</option>
                    </select>
                </td>
            </tr>
            <tr valign="top" class="custom_email_dependent_fields" style="display: none;">
                <th scope="row"><label for="custom_email_address">Recipient Email Address <span class="description">(required)</span> </label></th>
                <td><input type="text" name="custom_email_address" id="custom_email_address"><p class="description">Enter the email address which you want to be notified when status is trigger.</p></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="from_name">"From" Name <span class="description">(required)</span> </label></th>
                <td><input type="text" name="from_name" id="from_name"><p class="description">Enter the email name which will appear when the email is sent.</p></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="from_address">"From" Email Address <span class="description">(required)</span> </label></th>
                <td><input type="text" name="from_address" id="from_address"><p class="description">Enter the email address which will appear when the email is sent.</p></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="email_subject">Email Subject <span class="description">(required)</span> </label></th>
                <td><input type="text" name="email_subject" id="email_subject"><p class="description">Enter the email subject which will appear when the email is sent.</p></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="custom_title">Email Heading <span class="description">(required)</span> </label></th>
                <td><input type="text" name="custom_title" id="custom_title"><p class="description">Enter the heading which you want to appear on the email that is sent.</p></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="display_essential_info">Include Order Information?</label></th>
                <td><input type="checkbox" name="display_essential_info" id="display_essential_info" value="yes"><p class="description">Select whether you would like to include the order
information i.e. billing and shipping address, order items, total etc.</p></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="display_custom_info">Display Custom Message</label></th>
                <td><input type="checkbox" name="display_custom_info" id="display_custom_info" value="yes"></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;" id="custom_info_wrapper">
                <th scope="row">
                    <label for="custom_info">Message <span class="description">(required)</span></label>
                    <img class="help_tip pull_right" data-tip="Enter the message which will be sent in the email. You can use short codes." src="' . $woocommerce->plugin_url() . '/assets/images/help.png" height="16" width="16" />
                </th>
                <td>
                    <textarea name="custom_info" id="custom_info" rows="5" cols="40" style="float: left;"></textarea><br/><p style="float:left;" class="description">Available Shortcodes: {customer_first_name} , {customer_last_name} , {order_date} , {order_number} , {order_value} , {billing_address} , {shipping_address}.</p>
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="sends_email">Cancel Orders </label></th>
                <td>
                    <input type="checkbox" name="can_cancel" id="can_cancel" value="yes"><p class="description">Choose whether the customer can cancel orders when this status is applied.</p>
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="display_in_reports">Display In Reports? </label></th>
                <td>
                    <input type="checkbox" name="display_in_reports" id="display_in_reports" value="yes"><p class="description">Select wether you would like to include orders with status applied in reports as "Placed Orders".</p>
                </td>
            </tr>
            </tbody>
            </table>
            <h3>Action Settings</h3>
            <p>The following fields allows you to configure the Action icon that will trigger this custom status to the order which the Action icon is displayed on.</p>
            <table class="form-table">
            <tbody>
            <tr valign="top">
                <th scope="row"><label for="sends_email">Display Default Actions </label></th>
                <td>
                    <input type="checkbox" name="display_completed" id="display_completed" value="yes"> completed<p class="description">Check this to show the mark as completed button when this status is set</p>
                    <input type="checkbox" name="display_processing" id="display_processing" value="yes"> processing<p class="description">Check this to show the mark as processing button when this status is set</p>
                </td>
            </tr>
            <tr valign="top">
            <th scope="row"><label for="action_icon">Action Icon <span class="description">(required)</span></label></th>
            <td>';


                // Version 2.1
                ob_start();
                ?>
                <a class="button default-action-icon-selector tips woocommerce_status_action_font_icon woocommerce_status_actions_e17" href="javascript:;">Icon Preview</a>
                <input type="hidden" name="action_icon" value="e17" />
                <?php
                $display .= ob_get_clean();

            $display .='<a style="margin-left: 10px;" href="#TB_inline?width=600&height=550&inlineId=action-icons-list" class="thickbox">Click here to change the icon</a>
                <br><br><p class="description">Select an icon you would like to display as the Action button.</p>
            </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="display_for">Action Visibility <span class="description">(required)</span> </label></th>
                <td>';
                    foreach($status_list as $status_name){
                        $display .= "<input type='checkbox' class='display_for' name='display_for[]' id='display_for_{$status_name}' value='{$status_name}'> {$status_name}";
                        $display .= '<br>';
                    }
                    $display .= '
                    <p class="description">Select which statuses need to be applied to the order before this Action icon is shown.</p>
                </td>
            </tr>
</div>
</tbody>
</table>
<p>You can customise WooCommerce emails by clicking <a href="' . admin_url('admin.php?page=woocommerce_settings&tab=email') . '">here</a>. There you will find how to have more advanced controls on the layout and looks of your emails.</p>
<p class="submit"><input type="submit" name="submit" id="submit" class="button button-primary" value="Add New Status"></p>
</form>
</div>

<div id="action-icons-list" style="display:none">
    <div>
        <section id="web-application">
        <h3>Click on any of the following icons to set it</h2>
        ' . $this->generate_icons_list_from_img() . '
        </section>
    </div>
</div>

<div id="status-icons-list" style="display: none;">
    <div>
        <section id="web-application">
            <h3>Click on any of the following icons to set it</h2>
            ' . $this->generate_icons_list_from_img( true ) . '
        </section>
    </div>
</div>

';
        wp_enqueue_script(
            'validate_custom_status',
            plugin_dir_url( __FILE__ ) . '../js/validate_custom_status.js',
            array('jquery','jquery-validate'),
            '1.0',
            true
        );
        echo $display;
    }

    /**
     * Edit page HTML
     * @return void
     */
    function edit_status(){
        global $woocommerce;
        if(isset($_REQUEST['status_id'])){
            $display = '<div class="wrap">
            <div class="icon32"><img src="' . plugins_url( 'assets/img/status-actions-icon.png', dirname( __FILE__ ) ) . '" width="29" height="29" /></div>
    <h2>Edit Status</h2>
    <div class="error" style="display: none;">
        <p></p>
    </div>';
            try{
                if(isset($_REQUEST['action']) && $_REQUEST['action'] == 'save_edit'){
                    $woocommerce_status_actions = new WC_Custom_Status();
                    if($woocommerce_status_actions->add_status($_POST,true)){
                        // Display success message
                        $message = __('Status updated.','woocommerce_status_actions');
                        $display .= "<div id='message' class='updated below-h2'><p>{$message} <a href='".admin_url('admin.php?page=woocommerce_status_and_actions&tab=list_status') . "'></a></p></div>";
                    }
                }
                // Get already saved data to render in the HTML
                $status_name = WC_Custom_Status::get_status_name_for_id($_REQUEST['status_id']);
                if($status_name){
                    $woocommerce_status_actions = new WC_Custom_Status();
                    $woocommerce_status_actions->load_status_from_db($status_name);
                    $display_custom_info = '';
                    $display_essential_info = '';
                    if($woocommerce_status_actions->sends_email == '1'){
                        $sends_email = 'checked="checked"';
                        if($woocommerce_status_actions->display_custom_info == '1'){
                            $display_custom_info = 'checked="checked"';
                        }
                        if($woocommerce_status_actions->display_essential_info == '1'){
                            $display_essential_info = 'checked="checked"';
                        }
                    } else {
                        $sends_email = '';
                    }
                    $email_to_html = '';
                    $email_to = array('customer' => 'Customer','admin' => 'Admin','both'=>'Customer and Admin', 'custom' => 'Custom Email Address');
                    foreach($email_to as $email_key => $email_val){
                        if($email_key == $woocommerce_status_actions->email_to){
                            $email_to_html .= "<option value='{$email_key}' selected='selected'>{$email_val}</option>";
                        } else {
                            $email_to_html .= "<option value='{$email_key}'>{$email_val}</option>";
                        }
                    }


                    $status_color_type_html = '';
                    $status_color_types = array(
                            '1' => '<span style="background-color:'.$woocommerce_status_actions->status_color.';" class="old_status_icon status_edit woocommerce_status_font_icon woocommerce_status_actions_' . $woocommerce_status_actions->status_icon . '"></span>',
                            '0' => '<span style="border: 1px solid '. $woocommerce_status_actions->status_color .'; color:'. $woocommerce_status_actions->status_color .';" class="new_status_icon status_edit woocommerce_status_font_icon woocommerce_status_actions_' . $woocommerce_status_actions->status_icon . '"></span>',
                            '2' => '<strong class="status_text status_edit" style="background-color:'.$woocommerce_status_actions->status_color.';">'.$woocommerce_status_actions->status_name.'</strong>',
                            '3' => '<strong class="status_line_text status_edit" style="color:'.$woocommerce_status_actions->status_color.';">'.$woocommerce_status_actions->status_name.'</strong>'
                            );
                    foreach($status_color_types as $sc_key => $sc_val){
                        $checked = ($sc_key == $woocommerce_status_actions->status_color_type)?'checked':'';
                        $status_color_type_html .= "<label for='radio_$sc_key'><input type='radio' name='status_color_type' id='radio_$sc_key' value='$sc_key' $checked> $sc_val</label>";
                    }

                    $check_display_completed = ((!empty( $woocommerce_status_actions->display_completed ) ) && $woocommerce_status_actions->display_completed == '1') ? 'checked':'';
                    $check_display_processing = ((!empty( $woocommerce_status_actions->display_processing ) ) && $woocommerce_status_actions->display_processing == '1') ? 'checked':'';
                    $check_can_cancel = ((!empty( $woocommerce_status_actions->can_cancel ) ) && $woocommerce_status_actions->can_cancel == '1') ? 'checked':'';
                    $check_display_in_reports = ((!empty( $woocommerce_status_actions->display_in_reports ) ) && $woocommerce_status_actions->display_in_reports == '1') ? 'checked':'';

                    $action_icon = empty($woocommerce_status_actions->action_icon)?'alarm.png':$woocommerce_status_actions->action_icon;

                    $custom_status_list = WC_Custom_Status::get_status_list();
                    $status_list = array_merge($custom_status_list,WC_Custom_Status::$default_status_list);
                    add_thickbox();
                    $display .= '
    <form id="add_custom_status_form" method="POST" action="">
        <h3>Status Settings</h3>
        <p>The following fields allow you to configure the status to determine what happens when the status is applied to the order.</p>
        <input type="hidden" name="action" value="save_edit">
        <input type="hidden" name="status_id" value="'.strip_tags($_REQUEST['status_id']).'">
        <input type="hidden" name="term_id" value="'.$woocommerce_status_actions->term_id.'">
        <input type="hidden" name="term_taxonomy_id" value="'.$woocommerce_status_actions->term_taxonomy_id.'">
        <input type="hidden" name="status_name" value="'.$woocommerce_status_actions->status_name.'">
        <table class="form-table">
            <tbody>
            <tr valign="top">
                <th scope="row"><label for="status_name">Status Name <span class="description">(required)</span> </label></th>
                <td>'.$woocommerce_status_actions->status_name.'</td>
            </tr>
            <tr valign="top">
            <th scope="row"><label for="status_icon">Status Icon </label></th>
            <td>';

                if( empty( $woocommerce_status_actions->status_icon ) ) {
                    $woocommerce_status_actions->status_icon = 'alarm.png';
                }

                // Style!
                $style = '';
                if( empty( $woocommerce_status_actions->status_color_type ) || $woocommerce_status_actions->status_color_type == '0' ) {
                    $style = 'style="color:'. $woocommerce_status_actions->status_color .';font-size:1.4em;"';
                } else {
                    $style = 'style="background-color:'.$woocommerce_status_actions->status_color.';color:#fff;font-size:10px;border-radius:100%;padding:5px!important;margin-left:5px;';
                }

                ob_start();
                ?>
                <span style="padding: 0px 10px" class="default-status-icon-selector status_font_colour woocommerce_status_font_icon woocommerce_status_actions_<?php echo $woocommerce_status_actions->status_icon; ?>" href="javascript:;"></span>
                <input type="hidden" name="status_icon" value="<?php echo $woocommerce_status_actions->status_icon; ?>" />
                <?php
                $display .= ob_get_clean();


                $display .= '
                <a href="#TB_inline?width=600&height=550&inlineId=status-icons-list" class="thickbox">Click here to change the icon</a>
                <input type="hidden" name="status_icon" value="'. $woocommerce_status_actions->status_icon .'" />
                <p class="description">Select an icon you would like to display as the Status icon.</p>
            </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="status_color">Status Colour <span class="description">(required)</span> </label></th>
                <td>
                  <input type="text" name="status_color" id="status_color" value="'.$woocommerce_status_actions->status_color.'" data-default-color="#effeff" >
                  <p class="description">Select the main colour of your custom status.</p>
                </td>
            </tr>
            <tr valign="top">
                <th scope="row">
                  <label for="status_color_type">Status Style <span class="description">(required)</span></label>
                  <img class="help_tip pull_right" data-tip="Select a style for your custom status." src="' . $woocommerce->plugin_url() . '/assets/images/help.png" height="16" width="16" />
                </th>
                <td>
                        '.$status_color_type_html.'
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="sends_email">Email Notification </label></th>
                <td>
                  <input type="checkbox" name="sends_email" id="sends_email" value="yes" '.$sends_email.'>
                  <p class="description">Choose whether you would like to enable email notification when
this custom status is applied.</p>
                </td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="email_to">Recipient(s) <span class="description">(required)</span> </label>
                  <img class="help_tip pull_right" data-tip="Choose to whom you want the email to be sent to when the custom status is applied." src="' . $woocommerce->plugin_url() . '/assets/images/help.png" height="16" width="16" />
                </th>
                <td>
                    <select name="email_to" id="email_to">
                        '.$email_to_html.'
                    </select>
                </td>
            </tr>
            <tr valign="top" class="custom_email_dependent_fields" style="display: none;">
                <th scope="row"><label for="custom_email_address">Recipient Email Address <span class="description">(required)</span> </label></th>
                <td>
                    <input type="text" name="custom_email_address" id="custom_email_address" value="'.htmlentities(stripslashes($woocommerce_status_actions->custom_email_address)).'">
                    <p class="description">Enter the email address which you want to be notified when status is trigger.</p>
                </td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="from_name">"From" Name <span class="description">(required)</span> </label></th>
                <td>
                  <input type="text" name="from_name" id="from_name" value="'.htmlentities(stripslashes($woocommerce_status_actions->from_name)).'">
                  <p class="description">Enter the email name which will appear when the email is sent.</p>
                </td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="from_address">"From" Email Address <span class="description">(required)</span> </label></th>
                <td>
                  <input type="text" name="from_address" id="from_address" value="'.stripslashes($woocommerce_status_actions->from_address).'">
                  <p class="description">Enter the email address which will appear when the email is sent.</p>
                </td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="email_subject">Email Subject <span class="description">(required)</span> </label></th>
                <td>
                  <input type="text" name="email_subject" id="email_subject" value="'.htmlentities(stripslashes($woocommerce_status_actions->email_subject)).'">
                  <p class="description">Enter the email subject which will appear when the email is sent.</p>
                </td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="custom_title">Email Heading <span class="description">(required)</span> </label></th>
                <td>
                  <input type="text" name="custom_title" id="custom_title" value="'.htmlentities(stripslashes($woocommerce_status_actions->custom_title)).'">
                  <p class="description">Enter the heading which you want to appear on the email that is sent.</p>
                </td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="display_essential_info">Include Order Information?</label></th>
                <td>
                  <input type="checkbox" name="display_essential_info" id="display_essential_info" value="yes" '.$display_essential_info.'>
                  <p class="description">Select whether you would like to include the order
information i.e. billing and shipping address, order items, total etc.</p>
                </td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;">
                <th scope="row"><label for="display_custom_info">Display Custom Message</label></th>
                <td><input type="checkbox" name="display_custom_info" id="display_custom_info" value="yes" '.$display_custom_info.'></td>
            </tr>
            <tr valign="top" class="email_dependent_fields" style="display: none;" id="custom_info_wrapper">
                <th scope="row">
                  <label for="custom_info">Message <span class="description">(required)</span></label>
                  <img class="help_tip pull_right" data-tip="Enter the message which will be sent in the email. You can use short codes." src="' . $woocommerce->plugin_url() . '/assets/images/help.png" height="16" width="16" />
                  </th>
                <td>
                    <textarea name="custom_info" id="custom_info" rows="5" cols="40" style="float: left;">'.stripslashes($woocommerce_status_actions->custom_info).'</textarea><p style="float:left;" class="description">Available Shortcodes: {customer_first_name} , {customer_last_name} , {order_date} , {order_number} , {order_value} , {billing_address} , {shipping_address}.</p>
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="sends_email">Cancel Orders </label></th>
                <td>
                    <input type="checkbox" name="can_cancel" id="can_cancel" value="yes" '.$check_can_cancel.'><p class="description">Choose whether the customer can cancel orders when this status is applied.</p>
                </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="display_in_reports">Display In Reports? </label></th>
                <td>
                    <input type="checkbox" name="display_in_reports" id="display_in_reports" value="yes" '.$check_display_in_reports.'><p class="description">Select wether you would like to include orders with status applied in reports as "Placed Orders".</p>
                </td>
            </tr>
            </tbody>
            </table>
            <h3>Action Settings</h3>
            <p>The following fields allows you to configure the Action icon that will trigger this custom status to the order which the Action icon is displayed on.</p>
            <table class="form-table">
            <tbody>
            <tr valign="top">
                <th scope="row"><label for="sends_email">Display Default Actions </label></th>
                <td>
                    <input type="checkbox" name="display_completed" id="display_completed" value="yes" ' . $check_display_completed . ' > completed<p class="description">Check this to show the mark as completed button when this status is set</p>
                    <input type="checkbox" name="display_processing" id="display_processing" value="yes" ' . $check_display_processing . '> processing<p class="description">Check this to show the mark as processing button when this status is set</p>
                </td>
            </tr>
            <tr valign="top">
            <th scope="row"><label for="action_icon">Action Icon <span class="description">(required)</span></label></th>
            <td>';



                    // Version 2.1
                    ob_start();
                    ?>
                    <a class="button default-action-icon-selector tips woocommerce_status_action_font_icon woocommerce_status_actions_<?php echo $woocommerce_status_actions->action_icon; ?>" href="javascript:;">Icon Preview</a>
                    <input type="hidden" name="action_icon" value="<?php echo $woocommerce_status_actions->action_icon; ?>" />
                    <?php
                    $display .= ob_get_clean();

            $display .= '<a style="margin-left: 10px;" href="#TB_inline?width=600&height=550&inlineId=fontawesome-icons-list" class="thickbox">Click here to change the icon</a>
                </br><br><p class="description">Select an icon you would like to display as the Action button.</p>
            </td>
            </tr>
            <tr valign="top">
                <th scope="row"><label for="display_for">Action Visibility <span class="description">(required)</span> </label></th>
                <td>';
                   foreach($status_list as $status_name){
                        if( $status_name == $woocommerce_status_actions->status_name ) continue;
                        $checked = '';
                        if( !empty( $woocommerce_status_actions->display_for ) ) {
                            if(in_array($status_name,$woocommerce_status_actions->display_for)){
                                $checked = 'checked="checked"';
                            }
                        }
                        $display .= "<input type='checkbox' class='display_for' name='display_for[]' id='display_for_{$status_name}' value='{$status_name}' {$checked}> {$status_name}";
                        $display .= '<br>';
                    }
                    $display .= '
                    <p class="description">Select which statuses need to be applied to the order before this Action icon is shown.</p>
                </td>
            </tr>
            </tbody>
        </table>
        <p>You can customise WooCommerce emails by clicking <a href="' . admin_url('admin.php?page=woocommerce_settings&tab=email') . '">here</a>. There you will find how to have more advanced controls on the layout and looks of your emails.</p>
        <p class="submit submitbox">
            <input type="submit" name="submit" id="submit" class="button button-primary" value="Update Status">
            <a href="' . admin_url().'admin.php?page=woocommerce_status_and_actions&tab=delete_status&status_id='.(int)$woocommerce_status_actions->term_id .  '" class="submitdelete deletion">Delete Status</a>
        </p>
    </form>
</div>

<div id="fontawesome-icons-list" style="display:none">
    <div>
        <section id="web-application">
        <h3>Click on any of the following icons to set it</h2>
        ' . $this->generate_icons_list_from_img() . '
        </section>
    </div>
</div>
<div id="status-icons-list" style="display: none;">
    <div>
        <section id="web-application">
            <h3>Click on any of the following icons to set it</h2>
            ' . $this->generate_icons_list_from_img( true ) . '
        </section>
    </div>
</div>
';
                    wp_enqueue_script(
                        'validate_custom_status',
                        plugin_dir_url( __FILE__ ) . '../js/validate_custom_status.js',
                        array('jquery','jquery-validate'),
                        '1.0',
                        true
                    );
                }
            } catch (RequiredFieldMissingException $e){
                $message = __('Required fields ','woocommerce_status_actions');
                $message .= implode(',',$e->get_missing_fields());
                $message .= __(' missing','woocommerce_status_actions');
                $display .= "<div class='error'>{$message}</div>";
            } catch (WPErrorException $e){
                $message = implode(',',$e->get_error_messages());
                $display .= "<div class='error'>{$message}</div>";
            } catch (Exception $e){
                $message = $e->getMessage();
                $display .= "<div class='error'>{$message}</div>";
            }
            echo $display;
        }
    }

    /**
     * Displays the table that lists the statuses
     * @return void
     */
    function list_status(){
        $display = '';
        if( !empty( $_GET['success_add'] ) ) {
            $message = __('Status added successfully!', 'woocommerce_status_actions');
            $display = "<div id='message' class='updated below-h2'><p>{$message}</p></div>";
        }
        echo '<div class="wrap">
		<div class="icon32">
			<img src="' . plugins_url( 'assets/img/status-actions-icon.png', dirname( __FILE__ ) ) . '" width="29" height="29" />
		</div>
		<h2>Order Status & Actions <a href="'.admin_url().'admin.php?page=woocommerce_status_and_actions&tab=add_status" class="add-new-h2">Add Status</a></h2>';

        $woocommerce_status_actions_table = new WC_Custom_Status_Table();
        $woocommerce_status_actions_table->prepare_items();
        echo '<form action=" ' . admin_url(). 'admin.php?page=woocommerce_status_and_actions&tab=delete_status' . ' " method="POST">';
        $woocommerce_status_actions_table->display();
        echo '</form>';
        echo '</div>';
    }

    /**
     * Process the status changes
     * @param  [int] $order_id   [order id]
     * @param  [string] $old_status [old status]
     * @param  [string] $new_status [new status]
     * @return boolean
     */
    function process_order_status($order_id,$old_status,$new_status){
        $default_statuses = array('pending','failed','on-hold','processing','completed','refunded','cancelled');
        // Process the status if not one of the defaults
        if(!in_array($new_status,$default_statuses)){
            $woocommerce_status_actions = new WC_Custom_Status();
            try{
                $woocommerce_status_actions->process_status($order_id,$old_status,$new_status);
            } catch(Exception $e){

            }
        }
        return true;
    }


    /**
     * What data to view in the tables' columns
     * @param  [string] $column [column name]
     * @return void
     */
    function payment_column_manage($column){
        global $post, $woocommerce, $the_order;

        if ( empty( $the_order ) || $the_order->id != $post->ID )
            $the_order = new WC_Order( $post->ID );

        if( $column == 'order_status' ) {
            $status = new WC_Custom_Status();
            $status->load_status_from_db( $the_order->status );
            if( !empty( $status->status_name ) ) { // it's a custom one
                // Style!
                $style = '';
                $class = '';
                if( empty( $status->status_color_type ) || $status->status_color_type == '0' ) {
                    $style = 'style="border: 1px solid '. $status->status_color .'; color:'. $status->status_color .';"';
                    $class = 'new_status_icon';
                } else if( $status->status_color_type == '1' ){
                    $style = 'style="background-color: '.$status->status_color.';"';
                    $class = 'old_status_icon';

                }
                if( empty( $status->status_color_type ) || $status->status_color_type == '0' || $status->status_color_type == '1' ) {
                    echo '<span '.$style.'  class="default-status-icon-selector woocommerce_status_font_icon woocommerce_status_actions_'.$status->status_icon.' tips hhh '.$class.'" data-tip="'.$status->status_name.'" alt="'.$status->status_name.'"></span></td>';
                } else if( $status->status_color_type == '2' ){
                    echo '<strong class="status_orders_page status_text hhh" style="background-color:'.$status->status_color.';">'.$status->status_name.'</strong>';
                } else {
	                echo '<strong class="status_orders_page status_line_text hhh" style="color:'.$status->status_color.';">'.$status->status_name.'</strong>';
                }
                WC_Custom_Status_Plugin::hide_marks();
            } else {
                // It's probably a default
                $style = get_option( 'woocommerce_status_actions_default_style' );
                if( !empty( $style ) ) {
                    switch( $style ) {
                        case '1':
                            echo '<strong class="status_orders_page status_text status_text_'. $the_order->status .' hhh">'. $the_order->status .'</strong>';
                            WC_Custom_Status_Plugin::hide_marks();
                        break;
                        case '2':
                            echo '<strong class="status_orders_page status_line_text status_line_text_'. $the_order->status .' hhh">'. $the_order->status .'</strong>';
                            WC_Custom_Status_Plugin::hide_marks();
                        break;
                        case '0':
                        default:
                        break;
                    }
                }
            }
        } # end of order status
    }

    /**
     * Adds the action button of our custom status - per order
     * @param  [array] $actions   [actions for the order to be displayed]
     * @param  [object] $the_order [order object]
     * @return [array]            the actions array with our buttons added
     */
    function action_column_manage($actions, $the_order){
        global $woocommerce;

        $custom_action_list = WC_Custom_Status::get_status_list();
        foreach($custom_action_list as $custom_action){
            $s = new WC_Custom_Status();
            $s->load_status_from_db( $custom_action );
            $display_for = $s->display_for;
            if(in_array($the_order->status, $display_for)){
                $status = new WC_Custom_Status();
                $status->load_status_from_db( $custom_action );

                ########## Version ##############

                    // Version 2.1
                    $action_icon = empty($status->action_icon)?'e17':$status->action_icon;
                    $actions[$custom_action] = array(
                        'url'       => wp_nonce_url( admin_url( "admin-ajax.php?action=woocommerce-mark-order-{$custom_action}&order_id={$the_order->id}&custom_action_name={$custom_action}" ), "woocommerce-mark-order-{$custom_action}" ),
                        'name'      => __( ucfirst($custom_action), 'woocommerce_status_actions' ),
                        'action'    => 'woocommerce_status_action_font_icon woocommerce_status_actions_' . $action_icon
                    );
            }
            if( $the_order->status == $custom_action ) {
                if( (!empty($s->display_completed)) && $s->display_completed == '1' ) {

                    $actions['complete'] = array(
                        'url'       => wp_nonce_url( admin_url( 'admin-ajax.php?action=woocommerce_mark_order_complete&order_id=' . $the_order->id ), 'woocommerce-mark-order-complete' ),
                        'name'      => __( 'Complete', 'woocommerce' ),
                        'action'    => "complete"
                        );
                }
                if( (!empty($s->display_processing)) && $s->display_processing == '1' ) {
                    $actions['processing'] = array(
                            'url'       => wp_nonce_url( admin_url( 'admin-ajax.php?action=woocommerce_mark_order_processing&order_id=' . $the_order->id ), 'woocommerce-mark-order-processing' ),
                            'name'      => __( 'Processing', 'woocommerce' ),
                            'action'    => "processing"
                        );
                }
            }
        }
        return $actions;
    }

    /**
     * Generates the thickbox content that contains all action icons for the user to choose from
     * @return [string] [HTML to display]
     */
    function generate_icons_list_from_img( $status = false ) {
        global $woocommerce;
        $display = '';
        $class_selector = ($status)?'status-icon-selector':'action-icon-selector';

            // Version 2.1 Icons
            $font_file = plugin_dir_path( dirname(__FILE__) ) . 'css/fonts/icomoon.svg';
            if( file_exists( $font_file ) ) {
                $file = file_get_contents( $font_file );
                $matches = array();
                preg_match_all( "/glyph unicode=\"(.*?)\"/", $file, $matches);
                foreach( $matches[1] as $unicode ) {
                    $class_name = str_replace('&#x', '', $unicode);
                    $class_name = str_replace(';', '', $class_name);
                    ob_start();
                    ?>
                    <span class="<?php echo $class_selector; ?>-21 woocommerce_status_action_font_icon woocommerce_status_actions_<?php echo $class_name; ?>" data-src="<?php echo $class_name; ?>" href="javascript:;"></span>
                    <?php
                    $display .= ob_get_clean();
                }
            }
        return $display;
    }

    function add_general_settings_option( $settings ) {
        $status_actions_settings = array(
                        array(  'title' => __( 'Order Status Options', 'woocommerce' ), 'type' => 'title','desc' => '', 'id' => 'status_actions_options' ),
                        array(
                            'title' => __( 'Status Style', 'woocommerce' ),
                            'id'        => 'woocommerce_status_actions_default_style',
                            'default'   => '0',
                            'type'      => 'radio',
                            'desc_tip'  =>  __( 'This will effect the default WooCommerce\'s statuses (e.g. Processing, Completed, etc.).', 'woocommerce' ),
                            'options'   => array(
                                '0' => __( '<span class="settings-icon"></span>', 'woocommerce' ),
                                '1' => __( '<strong class="status_text">Background</strong>', 'woocommerce' ),
                                '2' => __( '<strong class="status_line_text">Text</strong>', 'woocommerce' )
                            ),
                        ),
                        array( 'type' => 'sectionend', 'id' => 'status_actions_options')
                    );
        return array_merge( $settings, $status_actions_settings );
    }

    /**
     * Checks that required fields exist
     * @param  [array] $required_fields [keys to look for]
     * @param  [array] $submit_method   [submitted data]
     * @return boolean
     */
    static function check_missing_fields($required_fields, $submit_method){

        $missing_fields = array();

        foreach($required_fields as $field_name){
            if(empty($submit_method[$field_name])){
                $missing_fields[] = $field_name;
            }
        }

        if(!empty($missing_fields)){
            throw new RequiredFieldMissingException($missing_fields);
        } else {
            return false;
        }
    }

    /**
     * Neat name to be displated in validation error messages
     * @param  [string] $slug [slug of the field]
     * @return [string]       [better name for the field]
     */
    static function get_field_name( $slug ) {
        switch ( $slug ) {
            case 'display_for':
                return 'action triggers';
            case 'status_name':
                return 'status name';
            default:
                return $slug;
        }
    }

    static function hide_marks() {
        ?>
        <script type="text/javascript">
            jQuery('.hhh').siblings('mark').hide();
        </script>
        <?php
    }

}
?>
