<?php
/*
Plugin Name: Woocommerce Product Colorizer
Plugin URI:
Description: Plugin generates live preview of different colors of products.
Version: 1.0.1
Author: OptArt | Piotr Szczygiel
Author URI: http://www.optart.biz
*/

add_action( 'init', 'init_wc_product_colorizer' );
function init_wc_product_colorizer()
{
    if ( class_exists( 'woocommerce' ) ) {
        new wc_product_colorizer();
    }
}

class wc_product_colorizer
{
    /**
     * Plugin identifier, it's mainly used for translations.
     */
    const PLUGIN_IDENTIFIER = 'wc-product-colorizer';

    /**
     * The slug for attribute, which is used by the plugin.
     */
    const ATTRIBUTE_SLUG = 'colorizer-color';

    /**
     * The default label for attribute, which is used by the plugin.
     */
    const ATTRIBUTE_LABEL = 'Product Colorizer Color';

    /**
     * The identifier of mask background image
     */
    const MASK_IMAGE_ID = 'colorizer_mask_image';

    /**
     * The identifier of colorizer background image
     */
    const BACKGROUND_IMAGE_ID = 'colorizer_background_image';

    /**
     * Contains the URL of default placeholder product image.
     * @var string
     */
    public static $def_image;

	/**
	 * Class default constructor
	 */
	public function __construct()
	{
        self::$def_image = woocommerce_placeholder_img_src();

        // hooks for colorizer meta box
        add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
        add_action( 'woocommerce_process_product_meta', array( $this, 'process_meta_box' ), 1, 1 );

        // Backend hooks
        add_action( 'created_term', array( $this, 'attribute_field_add_save' ), 10, 3 );
        add_action( 'edit_term', array( $this, 'attribute_field_add_save' ), 10, 3 );
        add_action( 'pa_' . self::ATTRIBUTE_SLUG . '_add_form_fields' , array( $this, 'attribute_field_add_display' ) );
        add_action( 'pa_' . self::ATTRIBUTE_SLUG . '_edit_form_fields' , array( $this, 'attribute_field_edit_display' ), 9, 1 );

        // Frontend hooks
        add_action( 'jigoshop_before_single_product_summary_thumbnails', array( $this, 'display_colorizer_images' ), 1, 2 );
        add_filter( 'woocommerce_single_product_image_html', array( $this, 'display_colorizer_images' ), 10, 2 );

        // assets hooks
        add_action( 'admin_enqueue_scripts', array( $this, 'add_assets_admin' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'add_assets_front' ) );
	}

    /**
     * Loading assets for admin panel
     */
    public function add_assets_admin()
    {
        // CSS
        wp_enqueue_style( 'colorizer_admin_styles', plugins_url( 'assets/css/admin.css', __FILE__ ) );

        // JS
        wp_enqueue_script( 'colorizer_admin_scripts', plugins_url( 'assets/js/admin.js', __FILE__ ), array( 'jquery', 'wp-color-picker' ) );
        wp_localize_script( 'colorizer_admin_scripts', 'php_data', array(
            'confirm'	=> __( 'Are you sure you want to remove colorizer image?', self::PLUGIN_IDENTIFIER ),
            'def_image'	=> self::$def_image,
            'colorpicker_id' => self::ATTRIBUTE_SLUG,
            'mask_id' => self::MASK_IMAGE_ID,
            'background_id' => self::BACKGROUND_IMAGE_ID
        ));
    }

    /**
     * Loading assets for landing page
     */
    public function add_assets_front()
    {
        if ( $this->is_colorizer_enabled() ) {
            // CSS
            wp_enqueue_style( 'colorizer_front_styles', plugins_url( 'assets/css/front.css', __FILE__ ) );

            // JS
            wp_enqueue_script( 'colorizer_lib', plugins_url( 'assets/js/jquery.productColorizer.js', __FILE__ ), array( 'jquery' ) );
            wp_enqueue_script( 'colorizer_front_scripts', plugins_url( 'assets/js/front.js', __FILE__ ), array( 'jquery' ) );
            wp_localize_script( 'colorizer_front_scripts', 'php_data', array(
                'colorizer_enabled' =>  $this->is_colorizer_enabled(),
                'attribute_slug' => self::ATTRIBUTE_SLUG
            ));
        }
    }

    /**
    * Displays meta boxes on product page in admin panel
    */
    public function add_meta_box()
    {
        add_meta_box( 'postcolorizerimagediv', __( 'Colorizer Mask Image' ), function( $post ) {
            $colorizer_image = wc_product_colorizer::get_colorizer_image_url(
                $post->ID,
                wc_product_colorizer::PLUGIN_IDENTIFIER,
                wc_product_colorizer::$def_image
            );
            $colorizer_id = wc_product_colorizer::get_colorizer_image($post->ID, wc_product_colorizer::PLUGIN_IDENTIFIER );
            $image_identifier = wc_product_colorizer::MASK_IMAGE_ID;

            // @param $colorizer_image
            // @param $colorizer_id
            // @param $image_identifier
            require( 'templates/meta-box.php' );
        }, 'product', 'side', 'low' );

        add_meta_box( 'postcolorizerbgkimagediv', __( 'Colorizer Background Image' ), function( $post ) {
            $colorizer_image = wc_product_colorizer::get_colorizer_image_url(
                $post->ID,
                wc_product_colorizer::BACKGROUND_IMAGE_ID,
                wc_product_colorizer::$def_image
            );
            $colorizer_id = wc_product_colorizer::get_colorizer_image( $post->ID, wc_product_colorizer::BACKGROUND_IMAGE_ID );
            $image_identifier = wc_product_colorizer::BACKGROUND_IMAGE_ID;

            // @param $colorizer_image
            // @param $colorizer_id
            // @param $image_identifier
            require( 'templates/meta-box.php' );
        }, 'product', 'side', 'low' );
    }

	/**
     * Displays the colocpicker input on add-attribute page
     */
    public function attribute_field_add_display()
    {
        $identifier = self::ATTRIBUTE_SLUG;
        $label = __( 'Color picker', self::PLUGIN_IDENTIFIER );
        $description = __( 'Pick the color which will be used for product image.', self::PLUGIN_IDENTIFIER );

        // @param string $identifier
        // @param string $label
        // @param string $description
        require ( 'templates/attribute-field-add.php' );
    }

    /**
     * Method is used for storing attribute additional data - hex color code
     * @param string $term_id
     * @param type $tt_id
     * @param string $taxonomy
     */
    public function attribute_field_add_save( $term_id, $tt_id, $taxonomy )
    {
        if ( $taxonomy !== 'pa_' . self::ATTRIBUTE_SLUG ) {
            return false;
        }

        update_metadata( 'woocommerce_term', $term_id, self::ATTRIBUTE_SLUG, filter_input( INPUT_POST, self::ATTRIBUTE_SLUG ) );
    }

    /**
     * Displays the colocpicker input on edit-attribute page
     * @param type $term
     */
    public function attribute_field_edit_display( $term )
    {
        $identifier = self::ATTRIBUTE_SLUG;
        $label = __( 'Color picker', self::PLUGIN_IDENTIFIER );
        $description = __( 'Pick the color which will be used for product image.', self::PLUGIN_IDENTIFIER );
        $value = $this->get_color_value( $term );

        // @param string $identifier
        // @param string $label
        // @param string $description
        // @param string $value
        require ( 'templates/attribute-field-edit.php' );
    }

    /**
     * Method returns the markup with base and mask colorizer images. Original markup is replaced when colorizer
     * is enabled for this product.
     * @param string $html
     * @param int $post_id
     * @return string
     */
    public function display_colorizer_images( $html, $post_id )
    {
        if ( !$this->is_colorizer_enabled() ) {
            return $html;
        }

        global $product;

        $colorizer_image = self::get_colorizer_image( $post_id, self::PLUGIN_IDENTIFIER );
        $image_title = esc_attr( get_the_title( get_post_thumbnail_id() ) );
        $image_link = wp_get_attachment_url( get_post_thumbnail_id() );
        $image_size = apply_filters( 'single_product_large_thumbnail_size', 'shop_single' );
        $image = get_the_post_thumbnail( $post_id, $image_size, array(
            'title' => $image_title
        ) );
        $attachment_count = count( $product->get_gallery_attachment_ids() );
        $gallery = $attachment_count > 0 ? '[product-gallery]' : '';
        $background_image = self::get_colorizer_image_url( $post_id, self::BACKGROUND_IMAGE_ID );

        $prod_attributes = woocommerce_get_product_terms( $post_id, 'pa_' . self::ATTRIBUTE_SLUG );

        if ( !empty( $prod_attributes ) ) {
            foreach ( $prod_attributes as $color ) {
                $hex_color = $this->get_rgb_from_term( $color );
                if ( !is_null( $hex_color ) ) {
                    $colors[$color->slug] = join( ',', $hex_color );
                }
            }
        }

        ob_start();
        // @param string $image_link
        // @param string $image_title
        // @param string $gallery
        // @param string $image
        // @param string $colorizer_image
        // @param string $image_size
        // @param array $colors
        // @param string $background_image
        require( 'templates/colorizer-images.php' );

        return ob_get_clean();
    }

	/**
	 * Method returns the hex value of term color, which is stored in database
	 * @param type $term
	 * @return string
	 */
	private function get_color_value( $term )
	{
		return get_metadata( 'woocommerce_term', $term->term_id, self::ATTRIBUTE_SLUG, true );
	}

    /**
     * Returns the object id of colorizer image or null
     * @param int $prod_id
     * @param string $meta_id
     * @return int
     */
    public static function get_colorizer_image( $prod_id, $meta_id )
    {
        $post_meta = get_post_meta( $prod_id, $meta_id );
        return isset( $post_meta[0] ) && filter_var( $post_meta[0], FILTER_VALIDATE_INT ) ? $post_meta[0] : '';
    }

    /**
     * Returns the URL of uploaded colorizer image
     * @param int $prod_id
     * @param string $meta_id
     * @param string $def
     * @return stirng
     */
    public static function get_colorizer_image_url( $prod_id, $meta_id, $def = '' )
    {
        $colorizer_image_id = self::get_colorizer_image( $prod_id, $meta_id );
        return !empty( $colorizer_image_id ) ? wp_get_attachment_url( $colorizer_image_id ) : $def;
    }

    /**
     * Method gets the term name and returns its rgb value as array
     * @param string $color
     * @return null|array
     */
    private function get_rgb_from_term( $term )
    {
        $rgb = null;

        if ( is_object( $term ) ) {
            $hex_color = $this->get_color_value( $term );
            if ( !empty( $hex_color ) ) {
                $rgb = sscanf( $hex_color, "#%02x%02x%02x" );
            }
        }
        return $rgb;
    }

    /**
    * Checks whether colorizer is enabled for current product
    * @global type $post
    * @return boolean
    */
    private function is_colorizer_enabled()
    {
        global $post;
        $colorizer_image = self::get_colorizer_image( $post->ID, self::PLUGIN_IDENTIFIER );
        // correct post object && product page && colorizer image for product not empty && existing thumbnail for product
        return ( !is_null( $post ) && is_product() && !empty( $colorizer_image ) && has_post_thumbnail() );
    }

    /**
     * Update colorizer images ids when updating the product
     * @param int $post_id
     * @param type $post
     */
    public function process_meta_box( $post_id )
    {
        update_post_meta( $post_id, self::PLUGIN_IDENTIFIER, intval( filter_input( INPUT_POST, self::MASK_IMAGE_ID) ) );
        update_post_meta( $post_id, self::BACKGROUND_IMAGE_ID, intval( filter_input( INPUT_POST, self::BACKGROUND_IMAGE_ID ) ) );
    }
}

// Hook into activation
register_activation_hook( __FILE__, function(){

	global $wpdb;

	// get the attribute row
	$row = $wpdb->get_row( "SELECT attribute_id FROM " . $wpdb->prefix . "woocommerce_attribute_taxonomies "
							. "WHERE attribute_name='" . wc_product_colorizer::ATTRIBUTE_SLUG . "'" );

	// in case when row doesn't exist, add one
	if ( is_null( $row ) ) {
        $attribute = array(
			'attribute_name' => wc_product_colorizer::ATTRIBUTE_SLUG,
			'attribute_label' => __( wc_product_colorizer::ATTRIBUTE_LABEL, wc_product_colorizer::PLUGIN_IDENTIFIER ),
			'attribute_type' => 'select',
            'attribute_orderby' => 'name'
        );

		$wpdb->insert( $wpdb->prefix . 'woocommerce_attribute_taxonomies', $attribute );
        do_action( 'woocommerce_attribute_added', $wpdb->insert_id, $attribute );
        delete_transient( 'wc_attribute_taxonomies' );
	}
});