<?php
/*
Plugin Name: WooCommerce Zapier Integration
Plugin URI: http://www.woothemes.com/products/woocommerce-zapier/
Description: Integrates WooCommerce with the <a href="http://www.zapier.com" target="_blank">Zapier</a> web automation service.
Version: 1.4
Author: OM4
Author URI: http://wczap.com/
Text Domain: wc_zapier
*/

/*
Copyright 2013-2014 OM4 (email: info@om4.com.au    web: http://om4.com.au/)

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * Required functions
 */
if ( ! function_exists( 'woothemes_queue_update' ) )
	require_once('woo-includes/woo-functions.php');

/**
 * Plugin updates
 */
woothemes_queue_update( plugin_basename( __FILE__ ), '0782bdbe932c00f4978850268c6cfe40', 243589 );

/**
 * Check WooCommerce exists and is active
 */
if ( is_woocommerce_active() ) {

	if ( ! class_exists( 'WC_Zapier' ) ) {

		/**
		 * This class is a singleton, and should be accessed via the WC_Zapier() function.
		 *
		 * Class WC_Zapier
		 */
		class WC_Zapier {

			/**
			 * Database version (used for install/upgrade tasks if required)
			 */
			const db_version = 2;

			/**
			 * Name of the wp_option record that stores the installed version number.
			 */
			const db_version_option_name = 'wc_zapier_version';

			/**
			 * The minimum WooCommerce version that this plugin supports.
			 */
			const MINIMUM_SUPPORTED_WOOCOMMERCE_VERSION = '2.1.0';

			/**
			 * URL to the documentation for this plugin.
			 */
			const documentation_url = 'http://docs.woothemes.com/document/woocommerce-zapier/';

			/**
			 * Stores the one and only instance of this class
			 * @var WC_Zapier
			 */
			private static $instance;

			/**
			 * Stores the WC_Zapier_Admin class
			 * @var WC_Zapier_Admin
			 */
			private $admin;

			/**
			 * Full path to this plugin's directory (including trailing slash)
			 * @var string
			 */
			public static $plugin_path;

			/**
			 * Full URL to this plugin's directory (including trailing slash)
			 * @var string
			 */
			public static $plugin_url;

			/**
			 * The list of WooCommerce order statuses.
			 *
			 * Generated by WC_Zapier::get_order_statuses()
			 *
			 * @var
			 */
			private static $order_statuses;

			/**
			 * The main WC_Zapier instance.
			 *
			 * @return WC_Zapier
			 */
			public static function instance() {
				if ( ! isset(self::$instance) ) {
					self::$instance = new WC_Zapier();
					self::$instance->initialise();
				}
				return self::$instance;
			}

			/**
			 * Initialise the plugin
			 */
			private function initialise() {

				self::$plugin_path = dirname( __FILE__ ) . '/';
				self::$plugin_url = plugin_dir_url( __FILE__ );

				// Set up class autoloading
				if ( function_exists( "__autoload" ) ) {
					spl_autoload_register( "__autoload" );
				}
				spl_autoload_register( array( $this, 'autoload' ) );

				load_plugin_textdomain( 'wc_zapier', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );

				add_action( 'plugins_loaded', array( $this, 'plugins_loaded' ) );

			}

			/**
			 * Autoload the WC_Zapier_* classes when required.
			 * Helps keep code simple and memory consumption down.
			 *
			 * @param string $class_name
			 */
			public function autoload( $class_name ) {

				// Only act on classes in this plugin
				if ( false === strpos( $class_name, 'WC_Zapier' ) )
					return;

				$filename = "{$class_name}.php";

				$directory = self::$plugin_path . 'includes/';

				if ( 'WC_Zapier_Trigger_Factory' !== $class_name && false !== strpos( $class_name, 'Zapier_Trigger_' ) )
					$directory .= 'triggers/';

				$file = $directory . $filename;

				if ( file_exists( $file ) )
					require_once( $file );

			}

			/**
			 * Executed during the 'plugins_loaded' WordPress hook.
			 *
			 * - Checks that we're running the correct WooCommerce Version
			 * - Sets up various hooks
			 * - Load Supported Zapier Triggers
			 * - Loads the admin/dashboard interface if required
			 */
			public function plugins_loaded() {

				// WooCommerce version check
				if ( version_compare( WOOCOMMERCE_VERSION, self::MINIMUM_SUPPORTED_WOOCOMMERCE_VERSION, '<' ) ) {
					add_action( 'admin_notices', array( $this, 'admin_notice' ) );
					WC_Zapier()->log( "WooCommerce plugin version(" . WOOCOMMERCE_VERSION . ") is less than " . self::MINIMUM_SUPPORTED_WOOCOMMERCE_VERSION );
					return;
				}

				// User has a supported version of WooCommerce, so let's proceed.

				// WooCommerce init is priority 0, cancel_order is priority 10
				// Initialize in between those two so that cancelled orders are accounted for
				add_action( 'init', array( $this, 'init' ), 5 );

				add_action( 'admin_init', array( $this, 'maybe_activate_or_update' ) );

				add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'action_links' ) );

				if ( function_exists( 'woocommerce_init_checkout_field_editor' ) ) {
					// The WooCommerce Checkout Field Editor plugin is active
					new WC_Zapier_Checkout_Field_Editor();
				}

				add_action( 'wc_zapier_resend_sample_data', array( 'WC_Zapier', 'resend_sample_data' ) );

			}

			/**
			 * Obtain the list of WooCommerce order statuses.
			 *
			 * @return string[] Array of order status slugs (excluding the wc- prefix)
			 */
			public static function get_order_statuses() {
				if ( is_null( self::$order_statuses ) ) {
					self::$order_statuses = array();
					if ( function_exists( 'wc_get_order_statuses' ) ) {
						// WC 2.2+
						$statuses = wc_get_order_statuses();
						foreach ( $statuses as $status => $status_label ) {
							// Use the order status without wc- internal prefix
							$status = 'wc-' === substr( $status, 0, 3 ) ? substr( $status, 3 ) : $status;
							self::$order_statuses[] = $status;
						}
					} else {
						// WC 2.1 and earlier
						$statuses = get_terms( 'shop_order_status', array( 'hide_empty' => 0, 'orderby' => 'id' ) );
						if ( ! is_wp_error( $statuses ) ) {
							foreach ( $statuses as $status ) {
								self::$order_statuses[] = $status->slug;
							}
						}
					}
				}
				return self::$order_statuses;
			}


			/**
			 * Displays a message if the user isn't using a supported version of WooCommerce.
			 */
			public function admin_notice() {
				?>
				<div id="message" class="error">
					<p><?php printf( __( 'The WooCommerce Zapier Integration plugin is only compatible with WooCommerce version %s or later. Please update WooCommerce.', 'wc_zapier' ), self::MINIMUM_SUPPORTED_WOOCOMMERCE_VERSION ); ?></p>
				</div>
			<?php
			}

			/**
			 * Registers the custom post type for storing Zapier feeds.
			 *
			 * Executed during the 'init' WordPress hook.
			 */
			public function init() {

				/*
				 * The Custom Post Type that stores the zapier feeds
				 */
				register_post_type(
					'wc_zapier_feed',
					array(
						'public'       => false
					, 'show_ui'      => true
					, 'show_in_menu' => 'woocommerce'
					, 'supports'     => false
					, 'labels'       => array(
							'name'         => __( 'Zapier Feeds', 'wc_zapier' )
						, 'singular_name'     => __( 'Zapier Feed', 'wc_zapier' )
						, 'add_new_item' => __( 'Add New Zapier Feed', 'wc_zapier' )
						, 'edit_item' => __( 'Edit Zapier Feed', 'wc_zapier' )
						, 'new_item' => __( 'New Zapier Feed', 'wc_zapier' )
						, 'view_item' => __( 'View Zapier Feed', 'wc_zapier' )
						, 'search_items' => __( 'Search Zapier Feeds', 'wc_zapier' )
						, 'not_found' => __( 'No Zapier Feeds found', 'wc_zapier' )
						, 'not_found_in_trash' => __( 'No Zapier Feeds found in Trash', 'wc_zapier' )
					)
					)
				);

				WC_Zapier_Trigger_Factory::load_triggers();

				if ( is_admin() ) {
					$this->admin = new WC_Zapier_Admin();
				}

			}

			/**
			 * Install the plugin if required.
			 *
			 * As per http://core.trac.wordpress.org/ticket/14170 it's far better to use an upgrade routine fired on admin_init
			 *
			 * Executed on every admin/dashboard page load (via the admin_init hook)
			*/
			public function maybe_activate_or_update() {
				if ( self::db_version !== $installed_version = intval( get_option( self::db_version_option_name ) ) ) {
					if ( 0 === $installed_version ) {
						// Plugin activation/installation tasks
					} else {
						// Any database update/upgrade routines will go here
						if ( 1 == $installed_version ) {
							// Send sample data to all active feeds to ensure they have the latest field definitions
							self::resend_sample_data_async();
						}
					}
					update_option( self::db_version_option_name, self::db_version );
				}
			}

			/**
			 * Return a formatted price (excluding currency symbols).
			 *
			 * Strangely, WooCommerce doesn't seem to have a native function for this.
			 *
			 * (woocommerce_price() includes the currency symbol, which we don't want).
			 *
			 * @param int|float|double $price The unformatted price
			 *
			 * @return string the formatted price (2 decimal places)
			 */
			public static function format_price( $price ) {
				return number_format( (double) $price, 2, '.', '' );
			}

			/**
			 * Format a WordPress date into a W3C formatted date (eg 2005-08-15T15:52:01+00:00)
			 *
			 * @param string $date WordPress date
			 *
			 * @return string date formatted like 2005-08-15T15:52:01+00:00
			 */
			public static function format_date( $date ) {
				// See http://www.php.net/manual/en/class.datetime.php#datetime.constants.types
				return date_i18n( DATE_W3C, strtotime( $date ) );
			}

			public function action_links( $links ) {

				$plugin_links = array(
					'<a href="' . admin_url( 'edit.php?post_type=wc_zapier_feed' ) . '">' . __( 'Settings', 'wc_zapier' ) . '</a>',
					'<a href="' . self::documentation_url . '">' . __( 'Documentation', 'wc_zapier' ) . '</a>'
				);

				return array_merge( $plugin_links, $links );
			}

			/**
			 * Asynchronously send sample data to all Feeds that use the specified trigger(s).
			 *
			 * This ensures that all existing Zaps include the latest data specification
			 *
			 * @param array $trigger_keys List of trigger keys (strings)
			 */
			public static function resend_sample_data_async( $trigger_keys = array() ) {
				wp_schedule_single_event( time(), 'wc_zapier_resend_sample_data', array( 'trigger_keys' => $trigger_keys ) );
			}

			/**
			 * Immediately re-send sample data to all currently active Zapier Feeds that use these triggers.
			 *
			 * This ensures that Zapier has the latest checkout field definitions.
			 *
			 * @param array $trigger_keys Array of trigger keys (strings)
			 */
			public static function resend_sample_data( $trigger_keys = array() ) {
				if ( empty($trigger_keys ) ) {
					$trigger_keys = WC_Zapier_Trigger_Factory::get_trigger_keys();
				}
				foreach ( $trigger_keys as $trigger_key ) {
					try {
						$trigger = WC_Zapier_Trigger_Factory::get_trigger_with_key( $trigger_key );
						$trigger->send_sample_data_to_active_feeds_using_this_trigger();
					} catch ( Exception $ex ) {
						// Invalid trigger key - silently ignore and continue
					}
				}
			}

			/**
			 * Log a message if the WC Zapier Debugger plugin is active.
			 *
			 * If the plugin isn't active, then no logging occurs
			 *
			 * @param string $message
			 * @param int|null $object_id Optional order ID or customer ID
			 * @param string|null Object Name. Order or Customer
			 */
			public function log( $message, $object_id = null, $object_name = 'Order' ) {
				if ( function_exists( "WC_Zapier_Debugger" ) ) {
					if ( !is_null( $object_id ) ) {
						$message = "$object_name #$object_id: $message";
					}

					WC_Zapier_Debugger()->log_message( $message );
				} else {
					// Debugger plugin not active. Do nothing.
				}
			}

		}

		/**
		 * This function should be used to access the WC_Zapier singleton class.
		 *
		 * It's simpler to use this function instead of a global variable.
		 *
		 * @return WC_Zapier
		 */
		function WC_Zapier() {
			return WC_Zapier::instance();
		}

		// Let's get this thing started!
		WC_Zapier();

	}

}